--[[
Copyright (C) GtX (Andy), 2019

Author: GtX | Andy
Date: 12.12.2019
Revision: FS25-02

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
]]

FillLevelLimiterDialog = {}

FillLevelLimiterDialog.NOT_APPLICABLE = "N/A"
FillLevelLimiterDialog.UPDATE_COUNTER_INTERVAL = 2000

FillLevelLimiterDialog.MOD_NAME = g_currentModName
FillLevelLimiterDialog.MOD_DIRECTORY = g_currentModDirectory

local FillLevelLimiterDialog_mt = Class(FillLevelLimiterDialog, YesNoDialog)

function FillLevelLimiterDialog.register()
    local dialog = FillLevelLimiterDialog.new()

    g_gui:loadGui(FillLevelLimiterDialog.MOD_DIRECTORY .. "gui/FillLevelLimiterDialog.xml", "FillLevelLimiterDialog", dialog)

    FillLevelLimiterDialog.INSTANCE = dialog
end

function FillLevelLimiterDialog.show(callback, target, title, vehicle, showHelpInfo)
    if FillLevelLimiterDialog.INSTANCE == nil then
        FillLevelLimiterDialog.register()

        return
    end

    local dialog = FillLevelLimiterDialog.INSTANCE

    dialog:setCallback(callback, target)
    dialog:setTitle(title)
    dialog:setData(vehicle, showHelpInfo)

    g_gui:showDialog("FillLevelLimiterDialog")
end

function FillLevelLimiterDialog.new()
    local self = YesNoDialog.new(nil, FillLevelLimiterDialog_mt)

    self.isButtonDisabled = false

    self.showHelpInfo = true
    self.doEnterPress = false

    self.updateCounter = FillLevelLimiterDialog.UPDATE_COUNTER_INTERVAL

    return self
end

function FillLevelLimiterDialog.createFromExistingGui(gui, guiName)
    FillLevelLimiterDialog.register()

    local callback = gui.callbackFunc
    local target = gui.target
    local title = gui.fillLevelLimiterTitle
    local vehicle = gui.vehicle
    local showHelpInfo = gui.showHelpInfo

    FillLevelLimiterDialog.show(callback, target, title, vehicle, showHelpInfo)
end

function FillLevelLimiterDialog:onGuiSetupFinished()
    FillLevelLimiterDialog:superClass().onGuiSetupFinished(self)

    local textInputElement = self.textInputFillLevel

    if textInputElement ~= nil and textInputElement.oldSetCaptureInput == nil then
        local oldSetCaptureInput = textInputElement.setCaptureInput

        textInputElement.setCaptureInput = function(element, isCapturing)
            oldSetCaptureInput(element, isCapturing)

            if element.isCapturingInput then
                element:setText("") -- Clear the text so it is easier to add a new value
            end
        end

        textInputElement.oldSetCaptureInput = oldSetCaptureInput
    end
end

function FillLevelLimiterDialog:update(dt)
    FillLevelLimiterDialog:superClass().update(self, dt)

    if self.vehicle ~= nil and self.fillUnitIndex ~= nil then
        self.updateCounter = self.updateCounter - dt

        if self.updateCounter < 0 then
            local fillUnits = self.vehicle:getFillUnits()
            local fillUnit = fillUnits ~= nil and fillUnits[self.fillUnitIndex]

            if fillUnit ~= nil and fillUnit.fillLevel ~= nil then
                if fillUnit.fillLevel ~= self.lastFillLevelForUpdate then
                    self.lastFillLevelForUpdate = fillUnit.fillLevel

                    local fillLevel = math.floor((fillUnit.fillLevel) + 0.5)
                    local capacity = math.floor((fillUnit.capacity or 0) + 0.5)
                    local unitText = fillUnit.unitText or g_i18n:getVolumeUnit(false)

                    self.textFillLevelTitle:setText(string.format(self.fillUnitHeader, math.max(fillLevel, math.min(1, capacity)), capacity, unitText))
                end
            end

            self.updateCounter = FillLevelLimiterDialog.UPDATE_COUNTER_INTERVAL
        end
    end
end

function FillLevelLimiterDialog:onOpen()
    FillLevelLimiterDialog:superClass().onOpen(self)

    self:updateAddRemoveButtons()
    FocusManager:setFocus(self.textInputFillLevel)
end

function FillLevelLimiterDialog:onClickOk()
    if self.isYesButtonDisabled then
        return true
    end

    if self.doEnterPress then
        self:onFillLevelEnterPressed()
    end

    self:sendCallback(self.fillLevelLimits, self.showHelpInfo)

    return false
end

function FillLevelLimiterDialog:onClickBack(forceBack, usedMenuButton)
    self:close()

    return false
end

function FillLevelLimiterDialog:onClickCancel()
    for fillUnitIndex, _ in pairs (self.fillLevelLimits) do
        local capacity = self.vehicle:getFillUnitCapacity(fillUnitIndex) or 0

        self.fillLevelLimits[fillUnitIndex] = capacity
        self.lastTextValue[fillUnitIndex] = tostring(capacity)
    end

    self.multiFillUnitIndex:setState(1, true)
    self:onClickOk()
end

function FillLevelLimiterDialog:onClickFillUnitIndex(state)
    local fillUnitIndex = self.fillUnitSelectorToIndex[state]
    local fillUnits = self.vehicle:getFillUnits()

    if fillUnitIndex ~= nil and fillUnits ~= nil then
        local fillUnit = fillUnits[fillUnitIndex]

        if fillUnit ~= nil then
            local fillLevel = math.floor((fillUnit.fillLevel or 1) + 0.5)
            local capacity = math.floor((fillUnit.capacity or 0) + 0.5)
            local unitText = fillUnit.unitText or g_i18n:getVolumeUnit(false)

            self.textFillLevelTitle:setText(string.format(self.fillUnitHeader, math.max(fillLevel, math.min(1, capacity)), capacity, unitText))

            self.fillUnitIndex = fillUnitIndex
            self.textInputFillLevel:setText(self:getCurrentTextValue())
        end
    end
end

function FillLevelLimiterDialog:onFillLevelEnterPressed()
    if self.textInputFillLevel.text ~= "" then
        local newFillLevelValue = tonumber(self.textInputFillLevel.text)

        if newFillLevelValue ~= nil then
            local fillLevel = self.vehicle:getFillUnitFillLevel(self.fillUnitIndex) or 1
            local capacity = self.vehicle:getFillUnitCapacity(self.fillUnitIndex) or 0

            -- self.fillLevelLimits[self.fillUnitIndex] = math.clamp(newFillLevelValue, math.max(math.floor(fillLevel + 0.5), 1), capacity)
            self.fillLevelLimits[self.fillUnitIndex] = math.min(math.max(newFillLevelValue, math.max(math.floor(fillLevel + 0.5), 1)), capacity)
        end
    end

    self.lastTextValue[self.fillUnitIndex] = string.format("%d", self.fillLevelLimits[self.fillUnitIndex])
    self.textInputFillLevel:setText(self.lastTextValue[self.fillUnitIndex])

    self:updateAddRemoveButtons()
    self.doEnterPress = false
end

function FillLevelLimiterDialog:onFillLevelEscPressed()
    self.textInputFillLevel:setText(self:getCurrentTextValue())
    self:updateAddRemoveButtons()
    self.doEnterPress = false
end

function FillLevelLimiterDialog:onFillLevelTextChanged()
    local text = self.textInputFillLevel.text

    if text ~= self.lastTextValue[self.fillUnitIndex] then
        if text ~= "" then
            if tonumber(text) ~= nil then
                self.lastTextValue[self.fillUnitIndex] = text
            else
                self.textInputFillLevel:setText(self.lastTextValue[self.fillUnitIndex])
            end
        end

        self.doEnterPress = true
    end
end

function FillLevelLimiterDialog:onClickAddRemove(button)
    local fillLevel = self.vehicle:getFillUnitFillLevel(self.fillUnitIndex) or 1
    local capacity = self.vehicle:getFillUnitCapacity(self.fillUnitIndex) or 0
    local newFillLevelValue = tonumber(self.textInputFillLevel.text) or 1

    if button == self.addRemoveButtons[1] then
        newFillLevelValue = math.max(math.floor(fillLevel + 0.5), 1)
    elseif button == self.addRemoveButtons[10] then
        newFillLevelValue = capacity
    else
        local value = tonumber(button.text)

        if value ~= nil then
            -- newFillLevelValue = math.clamp(newFillLevelValue + value, math.max(math.floor(fillLevel + 0.5), 1), capacity)
            newFillLevelValue = math.clamp(newFillLevelValue + value, math.max(math.floor(fillLevel + 0.5), 1), capacity)
        end
    end

    self.fillLevelLimits[self.fillUnitIndex] = newFillLevelValue
    self.lastTextValue[self.fillUnitIndex] = string.format("%d", newFillLevelValue)
    self.textInputFillLevel:setText(self.lastTextValue[self.fillUnitIndex])

    self:updateAddRemoveButtons()
end

function FillLevelLimiterDialog:onClickShowHelpInfo(state)
    self.showHelpInfo = state == BinaryOptionElement.STATE_RIGHT
end

function FillLevelLimiterDialog:focusChangeOverride(target, direction)
    if direction == FocusManager.TOP then
        local element = self.addRemoveButtons[10]

        if element:getIsDisabled() then
            element = self.addRemoveButtons[5]
        end

        return true, element
    end

    return nil, nil
end

function FillLevelLimiterDialog:sendCallback(fillLevelLimits, showHelpInfo)
    if self.inputDelay < self.time then
        self:close()

        if self.callbackFunc ~= nil then
            if self.target ~= nil then
                self.callbackFunc(self.target, fillLevelLimits, showHelpInfo)
            else
                self.callbackFunc(fillLevelLimits)
            end
        end
    end
end

function FillLevelLimiterDialog:updateAddRemoveButtons()
    local fillLevel = math.max(self.vehicle:getFillUnitFillLevel(self.fillUnitIndex) or 0, 1)
    local capacity = self.vehicle:getFillUnitCapacity(self.fillUnitIndex) or 0
    local fillLevelLimit = self.fillLevelLimits[self.fillUnitIndex] or (capacity - 1)

    for i, buttonElement in ipairs (self.addRemoveButtons) do
        if i > 5 then
            buttonElement:setDisabled(fillLevelLimit >= capacity)
        else
            buttonElement:setDisabled(fillLevelLimit <= fillLevel)
        end
    end
end

function FillLevelLimiterDialog:setData(vehicle, showHelpInfo)
    self.vehicle = vehicle
    self.fillUnitIndex = nil

    self.lastTextValue = {}
    self.fillLevelLimits = {}

    self.fillUnitSelectorToIndex = {}
    self.fillUnitHeader = g_i18n:getText("ui_fll_fillUnitHeader", FillLevelLimiterDialog.MOD_NAME)

    local fillUnitSelectorNames = {}
    local spec = vehicle.spec_fillLevelLimiter --vehicle[FillLevelLimiter.SPEC_TABLE_NAME]

    local resetDisabled = true

    if spec ~= nil then
        for fillUnitIndex, limit in pairs (spec.fillUnitIndexToLimit) do
            if limit < (self.vehicle:getFillUnitCapacity(fillUnitIndex) or 0) then
                resetDisabled = false
            end

            self.lastTextValue[fillUnitIndex] = tostring(limit)
            self.fillLevelLimits[fillUnitIndex] = limit

            table.insert(self.fillUnitSelectorToIndex, fillUnitIndex)
            table.insert(fillUnitSelectorNames, spec.fillUnitTitle[fillUnitIndex] or FillLevelLimiterDialog.NOT_APPLICABLE)
        end
    end

    self.multiFillUnitIndex:setTexts(fillUnitSelectorNames)
    self.multiFillUnitIndex:setState(1, true)

    self.showHelpInfo = Utils.getNoNil(showHelpInfo, true)
    self.binaryShowHelpInfo:setIsChecked(self.showHelpInfo, true, false)

    self:setYesButtonDisabled(spec == nil)
    self:setResetButtonDisabled(resetDisabled)
end

function FillLevelLimiterDialog:setTitle(title)
    self.fillLevelLimiterTitle = title or FillLevelLimiterDialog.NOT_APPLICABLE
    FillLevelLimiterDialog:superClass().setTitle(self, self.fillLevelLimiterTitle)
end

function FillLevelLimiterDialog:setYesButtonDisabled(disabled)
    self.isYesButtonDisabled = disabled
    self.yesButton:setDisabled(disabled)
end

function FillLevelLimiterDialog:setResetButtonDisabled(disabled)
    self.isResetButtonDisabled = disabled
    self.resetButton:setDisabled(disabled)
end

function FillLevelLimiterDialog:getCurrentTextValue()
    if self.fillUnitIndex ~= nil and self.fillLevelLimits[self.fillUnitIndex] ~= nil then
        return string.format("%d", self.fillLevelLimits[self.fillUnitIndex])
    end

    return ""
end
