--[[
	ToggleableBaleWrapper.lua
	
	Autor: 		Ifko[nator]
	Datum: 		28.08.2025
	Version: 	1.1
	
	Changelog:	v1.0 @24.11.2024 - initial implementation in FS 25
				---------------------------------------------------------------------------------------------------------------------------------------------------------------------
				v1.1 @08.03.2025 - fixed issue, that blocked the vehicle controls, if the drop mode was on "collect" or "manually" and a bale was on the bale wrapper
								 - baler with bale wrappers will now consume bale foil
								 - added hud icon for bale wrapper status for baler with bale wrappers
				----------------------------------------------------------------------------------------------------------------------------------------------------------------------
				v1.2 @28.08.2025 - fixed issue, that blocked the bale wrapping for forage, chaff and sugar beet cut bales with the goeweil vario master. 
								   fix for following error: .../FS25_roundBalerExtension/specializations/ToggleableBaleWrapper.lua:331: attempt to index nil with 'baleWrapperIsActive'
]]

ToggleableBaleWrapper = {};

ToggleableBaleWrapper.modName = "";
ToggleableBaleWrapper.currentModDirectory = "";

ToggleableBaleWrapper.DROP_MODE_MANUAL = 1;
ToggleableBaleWrapper.DROP_MODE_AUTOMATIC = 2;
ToggleableBaleWrapper.DROP_MODE_COLLECT = 3;
ToggleableBaleWrapper.DROP_MODE_MAX_STATE = 3;

ToggleableBaleWrapper.FORCE_DROP_BALE = false;

ToggleableBaleWrapper.HUD = {};
ToggleableBaleWrapper.HUD.COLOR = {};

ToggleableBaleWrapper.HUD.COLOR.OFF = {1, 1, 1, 1};
ToggleableBaleWrapper.HUD.COLOR.NOT_AVAILABLE = {0.7012, 0.0143, 0.0078, 1};

for _, mod in pairs(g_modManager.mods) do
	if mod.title:upper() == "RUNDBALLENPRESSEN ERWEITERUNG" or mod.title:upper() == "ROUND BALER EXTENSION" then		
		if g_modIsLoaded[tostring(mod.modName)] then	
			ToggleableBaleWrapper.modName = mod.modName;
			ToggleableBaleWrapper.currentModDirectory = mod.modDir;

			break;
		end;
	end;
end;

function ToggleableBaleWrapper.initSpecialization()
	local schemaSavegame = Vehicle.xmlSchemaSavegame;

	schemaSavegame:register(XMLValueType.INT, "vehicles.vehicle(?).toggleableBaleWrapper#currentDropModeState", "Current drop mode.", ToggleableBaleWrapper.DROP_MODE_MANUAL);
	schemaSavegame:register(XMLValueType.BOOL, "vehicles.vehicle(?).toggleableBaleWrapper#baleWrapperIsActive", "Is bale wrapper active.", true);
end;

function ToggleableBaleWrapper.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Baler, specializations) and SpecializationUtil.hasSpecialization(BaleWrapper, specializations);
end;

function ToggleableBaleWrapper.registerEventListeners(vehicleType)
	local functionNames = {
		"onLoad",
		"onUpdate",
		"saveToXMLFile",
		"onWriteStream",
		"onReadStream",
		"onRegisterActionEvents"
	};

	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, ToggleableBaleWrapper);
	end;
end;

function ToggleableBaleWrapper.registerFunctions(vehicleType)
	local functionNames = {
		"setBaleWrapperDropMode",
		"setBaleWrapperState"
	};

	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerFunction(vehicleType, functionName, ToggleableBaleWrapper[functionName]);
	end;
end;

local function scalePixelValuesToScreenVector(uiScale, iconX, iconY)
	return iconX * uiScale * g_aspectScaleX / g_referenceScreenWidth, iconY * uiScale * g_aspectScaleY / g_referenceScreenHeight;
end;

function ToggleableBaleWrapper:onLoad(savegame)
	local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
	local specBaleWrapper = self.spec_baleWrapper;

	specToggleableBaleWrapper.isGoeweilBaler = Utils.getNoNil(self.configFileName:find("varioMaster.xml") or self.configFileName:find("ltMaster.xml"), false);

	specToggleableBaleWrapper.baleWrapperIsActive = true;
	specToggleableBaleWrapper.allowSetBaleWrapperState = true;

	specToggleableBaleWrapper.currentDropModeState = ToggleableBaleWrapper.DROP_MODE_MANUAL;
	specToggleableBaleWrapper.maxDropState = ToggleableBaleWrapper.DROP_MODE_MAX_STATE;

	if self.spec_baler.nonStopBaling then
		specToggleableBaleWrapper.maxDropState = 2;
	end;

	specToggleableBaleWrapper.l10NTexts = {};

	if specBaleWrapper ~= nil then
		if savegame ~= nil and not savegame.resetVehicles then
			specToggleableBaleWrapper.baleWrapperIsActive = savegame.xmlFile:getValue(savegame.key .. ".toggleableBaleWrapper#baleWrapperIsActive", specToggleableBaleWrapper.baleWrapperIsActive);
			specToggleableBaleWrapper.currentDropModeState = savegame.xmlFile:getValue(savegame.key .. ".toggleableBaleWrapper#currentDropModeState", specToggleableBaleWrapper.currentDropModeState);
		end;

		if not specToggleableBaleWrapper.isGoeweilBaler then
			self:setBaleWrapperDropMode(specToggleableBaleWrapper.currentDropModeState, false);
		end;

		self:setBaleWrapperState(specToggleableBaleWrapper.baleWrapperIsActive, false);

		self.doStateChange = Utils.overwrittenFunction(self.doStateChange, ToggleableBaleWrapper.doStateChange);
		self.pickupWrapperBale = Utils.overwrittenFunction(self.pickupWrapperBale, ToggleableBaleWrapper.pickupWrapperBale);

		local l10NTexts = {
			"BALE_WRAPPER_MODE_ACTIVATED",
			"BALE_WRAPPER_MODE_DEACTIVATED",
			"DROP_MODE_MANUALLY",
			"DROP_MODE_AUTOMATIC",
			"DROP_MODE_COLLECT",
			"DROP_MODE_CURRENT",
		};

		for _, l10NText in pairs(l10NTexts) do
			specToggleableBaleWrapper.l10NTexts[l10NText] = g_i18n:getText(l10NText, ToggleableBaleWrapper.modName);
		end;

		specToggleableBaleWrapper.baleWrapperIconFilename = Utils.getFilename("huds/hud_baleWrapper.png", ToggleableBaleWrapper.currentModDirectory);
		specToggleableBaleWrapper.baleWrapperNotAvailableIconFilename = Utils.getFilename("huds/hud_baleWrapperNotAvailable.png", ToggleableBaleWrapper.currentModDirectory);

    	if specToggleableBaleWrapper.baleWrapperIconFilename ~= nil and specToggleableBaleWrapper.baleWrapperNotAvailableIconFilename ~= nil then
    	    local uiScale = g_gameSettings:getValue("uiScale");

    	    local width, height = scalePixelValuesToScreenVector(uiScale, 30, 30);
    	    local iconX, iconY = scalePixelValuesToScreenVector(uiScale, 1848, 34);
		
    	    specToggleableBaleWrapper.iconX = iconX
    	    specToggleableBaleWrapper.iconY = iconY
		
    	    specToggleableBaleWrapper.baleWarpperOverlay = Overlay.new(specToggleableBaleWrapper.baleWrapperIconFilename, 0, 0, width, height);
    	    specToggleableBaleWrapper.baleWarpperNotAvailableOverlay = Overlay.new(specToggleableBaleWrapper.baleWrapperNotAvailableIconFilename, 0, 0, width, height);
		
    	    specToggleableBaleWrapper.baleWarpperOverlay:setDimension(width, height);
    	    specToggleableBaleWrapper.baleWarpperNotAvailableOverlay:setDimension(width, height);
		
    	    specToggleableBaleWrapper.uiScale = uiScale;
    	end;
	end;
end;

function ToggleableBaleWrapper:draw()
    local vehicle = self.vehicle;

    if vehicle == nil or not self.isVehicleDrawSafe then
        return;
    end;

    local posX, posY = self:getPosition();
   
    local specAttacherJoints = vehicle.spec_attacherJoints;
	
	if specAttacherJoints ~= nil then
		for _, implement in pairs(vehicle:getAttachedImplements()) do
            local specToggleableBaleWrapper = implement.object.spec_toggleableBaleWrapper;

            if specToggleableBaleWrapper ~= nil and specToggleableBaleWrapper.baleWarpperOverlay ~= nil and specToggleableBaleWrapper.baleWarpperNotAvailableOverlay ~= nil then
                local currentColor = ToggleableBaleWrapper.HUD.COLOR.OFF;

                local uiScale = g_gameSettings:getValue("uiScale");

                if specToggleableBaleWrapper.uiScale ~= uiScale then
                    local width, height = scalePixelValuesToScreenVector(uiScale, 30, 30);
                   
                    specToggleableBaleWrapper.baleWarpperOverlay:setDimension(width, height);
                    specToggleableBaleWrapper.baleWarpperNotAvailableOverlay:setDimension(width, height);

                    specToggleableBaleWrapper.uiScale = uiScale;
                end;

				local currentBaleWarpperOverlay = specToggleableBaleWrapper.baleWarpperOverlay;
            
                if not specToggleableBaleWrapper.allowSetBaleWrapperState then
					currentColor = ToggleableBaleWrapper.HUD.COLOR.NOT_AVAILABLE;

					currentBaleWarpperOverlay = specToggleableBaleWrapper.baleWarpperNotAvailableOverlay;
				elseif specToggleableBaleWrapper.baleWrapperIsActive then
                    currentColor = HUD.COLOR.ACTIVE;
                end;
            
                currentBaleWarpperOverlay:setColor(unpack(currentColor));
				currentBaleWarpperOverlay:setPosition(posX + (self.aiIconOffsetX * 1.26), posY + self.aiIconOffsetY);
                currentBaleWarpperOverlay:render();
            end;
        end;
    end;
end;

SpeedMeterDisplay.draw = Utils.appendedFunction(SpeedMeterDisplay.draw, ToggleableBaleWrapper.draw);

function ToggleableBaleWrapper:setBaleWrapperDropMode(currentDropModeState, noEventSend)
    local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;

	if currentDropModeState ~= specToggleableBaleWrapper.currentDropModeState then
		specToggleableBaleWrapper.currentDropModeState = currentDropModeState;

		if not noEventSend then
			if g_server ~= nil then
				g_server:broadcastEvent(SetBaleWrapperDropModeEvent.new(self, currentDropModeState), nil, nil, self);
			else
				g_client:getServerConnection():sendEvent(SetBaleWrapperDropModeEvent.new(self, currentDropModeState));
			end;
		end;
	end;
end;

function ToggleableBaleWrapper:onWriteStream(streamId, connection)
	if not connection:getIsServer() then 
		local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
		local specBaleWrapper = self.spec_baleWrapper;

		if specBaleWrapper ~= nil then	
			streamWriteUIntN(streamId, specToggleableBaleWrapper.currentDropModeState, specToggleableBaleWrapper.maxDropState);
			streamWriteBool(streamId, specToggleableBaleWrapper.baleWrapperIsActive);
		end;
	end;
end;

function ToggleableBaleWrapper:onReadStream(streamId, connection)
	if connection:getIsServer() then
		local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
		local specBaleWrapper = self.spec_baleWrapper;

		if specBaleWrapper ~= nil then
			specToggleableBaleWrapper.currentDropModeState = streamReadUIntN(streamId, specToggleableBaleWrapper.maxDropState);
			specToggleableBaleWrapper.baleWrapperIsActive = streamReadBool(streamId);
		end;
	end;
end;

function ToggleableBaleWrapper:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
	if self.isClient then
        local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
		local specBaleWrapper = self.spec_baleWrapper;
		
		self:clearActionEventsTable(specToggleableBaleWrapper.actionEvents);

		if self:getIsActiveForInput(true, false) and specBaleWrapper ~= nil then
            local newFunctions = {
				"CHANGE_DROP_MODE_BUTTON",
				"TOGGLE_BALE_WRAPPER_BUTTON"
			};

			for _, newFunction in ipairs(newFunctions) do
				local allowRegisterInput = true;

				if specToggleableBaleWrapper.isGoeweilBaler then
					allowRegisterInput = newFunction ~= "CHANGE_DROP_MODE_BUTTON";
				else
					if self.spec_baleWrapper ~= nil then
						actionEventAutomaticDropActivation = self.spec_baleWrapper.actionEvents[InputAction.IMPLEMENT_EXTRA4];
						actionEventManualDrop = self.spec_baleWrapper.actionEvents[InputAction.IMPLEMENT_EXTRA3];

						if actionEventAutomaticDropActivation ~= nil then
							--## deactivate default button for switching between automatic and manual unload for balers with bale wrappers
							
							g_inputBinding:setActionEventActive(actionEventAutomaticDropActivation.actionEventId, false);
						end;

						if actionEventManualDrop ~= nil then
							--## deactivate default button for manual drop if the collect mode is active
							
							g_inputBinding:setActionEventActive(actionEventManualDrop.actionEventId, specBaleWrapper.baleWrapperState == BaleWrapper.STATE_WRAPPER_FINSIHED and specToggleableBaleWrapper.currentDropModeState == ToggleableBaleWrapper.DROP_MODE_MANUAL);
						end;
					end;
				end;

				if allowRegisterInput then
					local _, actionEventId = self:addActionEvent(specToggleableBaleWrapper.actionEvents, InputAction[newFunction], self, ToggleableBaleWrapper.actionEventChangeDropeModeAndBaleWrapperState, false, true, false, true, nil);

					g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_VERY_HIGH);
					g_inputBinding:setActionEventActive(actionEventId, newFunction == "CHANGE_DROP_MODE_BUTTON");
				end;

				ToggleableBaleWrapper.updateActionEvents(self);
			end;
		end;
	end;
end;

function ToggleableBaleWrapper.actionEventChangeDropeModeAndBaleWrapperState(self, actionName, inputValue, callbackState, isAnalog)
	local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;

	if actionName == "CHANGE_DROP_MODE_BUTTON" then
		if specToggleableBaleWrapper.currentDropModeState < specToggleableBaleWrapper.maxDropState then
			self:setBaleWrapperDropMode(specToggleableBaleWrapper.currentDropModeState + 1);
		else
			self:setBaleWrapperDropMode(ToggleableBaleWrapper.DROP_MODE_MANUAL);
		end;
	else
		self:setBaleWrapperState(not specToggleableBaleWrapper.baleWrapperIsActive, false);
	end;
end;

function ToggleableBaleWrapper:baleWrapperOnLoadFinished(superFunc, savegame)
	local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
    local specBaleWrapper = self.spec_baleWrapper;

    if specBaleWrapper.baleToLoad ~= nil then
        local baleToLoad = specBaleWrapper.baleToLoad;

        specBaleWrapper.baleToLoad = nil;

        local baleObject = Bale.new(self.isServer, self.isClient);
        local baleTransX, baleTransY, baleTransZ = unpack(baleToLoad.translation);
        local baleRotX, baleRotY, baleRotZ = unpack(baleToLoad.rotation);

        if baleObject:loadFromConfigXML(baleToLoad.filename, baleTransX, baleTransY, baleTransY, baleRotX, baleRotY, baleRotZ, baleToLoad.attributes.uniqueId) then
            baleObject:applyBaleAttributes(baleToLoad.attributes);
            baleObject:register();

            if baleObject.nodeId ~= nil and baleObject.nodeId ~= 0 then
				local baleId = NetworkUtil.getObjectId(baleObject);
				local bale = NetworkUtil.getObject(baleId);
				local baleWrapperIsActive = true;

				if specToggleableBaleWrapper ~= nil then
					baleWrapperIsActive = specToggleableBaleWrapper.baleWrapperIsActive;
				end;
                
				self:doStateChange(BaleWrapper.CHANGE_GRAB_BALE, baleId);
                self:doStateChange(BaleWrapper.CHANGE_DROP_BALE_AT_GRABBER);

				if bale:getSupportsWrapping() and baleWrapperIsActive then				
                	--## GIANTS.. Only update the wrapping state after loding, when the bale on the wrapper warpping supports..
					
					self:doStateChange(BaleWrapper.CHANGE_WRAPPING_START);

                	specBaleWrapper.currentWrapper.currentTime = baleToLoad.wrapperTime;

                	local wrapperState = math.min(baleToLoad.wrapperTime / specBaleWrapper.currentWrapper.animTime, 1);

                	baleObject:setWrappingState(wrapperState);

                	local wrapAnimation = specBaleWrapper.currentWrapper.animations.wrapBale;
                	local wrappingTime = specBaleWrapper.currentWrapper.currentTime / specBaleWrapper.currentWrapper.animTime;

                	self:updateWrappingState(wrappingTime);

                	AnimatedVehicle.updateAnimations(self, 99999999, true);

                	if wrappingTime < 1 then
                	    self:setAnimationTime(wrapAnimation.animName, wrappingTime, true);

                	    self:playAnimation(wrapAnimation.animName, wrapAnimation.animSpeed, wrappingTime, true);
                	else
                	    self:doStateChange(BaleWrapper.CHANGE_WRAPPING_BALE_FINSIHED);
                	    self:setAnimationTime(wrapAnimation.animName, wrappingTime, true);

                	    specBaleWrapper.setWrappingStateFinished = false;
                	end;
				end;
            end;
        end;
    end;
end;

BaleWrapper.onLoadFinished = Utils.overwrittenFunction(BaleWrapper.onLoadFinished, ToggleableBaleWrapper.baleWrapperOnLoadFinished);

function ToggleableBaleWrapper:doStateChange(superFunc, id, nearestBaleServerId)
    local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
	local specBaleWrapper = self.spec_baleWrapper;

	if id == BaleWrapper.CHANGE_WRAPPING_START or specBaleWrapper.baleWrapperState ~= BaleWrapper.STATE_WRAPPER_FINSIHED and id == BaleWrapper.CHANGE_WRAPPER_START_DROP_BALE then
		local bale = NetworkUtil.getObject(specBaleWrapper.currentWrapper.currentBale);
		local baleType = specBaleWrapper.currentWrapper.allowedBaleTypes[specBaleWrapper.currentBaleTypeIndex];

		if not bale:getSupportsWrapping() or baleType.skipWrapping or bale.wrappingState == 1 or not specToggleableBaleWrapper.baleWrapperIsActive then
			if self.isServer then
				specBaleWrapper.setWrappingStateFinished = true;
			end;

			return;
		end;
	end;

	if id == BaleWrapper.CHANGE_GRAB_BALE then
		local bale = NetworkUtil.getObject(nearestBaleServerId);

		specBaleWrapper.baleGrabber.currentBale = nearestBaleServerId;

		if bale ~= nil then
			local x, y, z = localToLocal(bale.nodeId, getParent(specBaleWrapper.baleGrabber.grabNode), 0, 0, 0);

			setTranslation(specBaleWrapper.baleGrabber.grabNode, x, y, z);

			bale:mountKinematic(self, specBaleWrapper.baleGrabber.grabNode, 0, 0, 0, 0, 0, 0);
			bale:setCanBeSold(false);
			bale:setNeedsSaving(false);

			specBaleWrapper.baleToMount = nil;

			self:playMoveToWrapper(bale);
		else
			specBaleWrapper.baleToMount = {
				serverId = nearestBaleServerId,
				linkNode = specBaleWrapper.baleGrabber.grabNode,
				trans = {0, 0, 0},
				rot = {0, 0, 0}
			};
		end;

		specBaleWrapper.baleWrapperState = BaleWrapper.STATE_MOVING_BALE_TO_WRAPPER;
		
	elseif id == BaleWrapper.CHANGE_DROP_BALE_AT_GRABBER then
		local attachNode = specBaleWrapper.currentWrapper.baleNode;
		local bale = NetworkUtil.getObject(specBaleWrapper.baleGrabber.currentBale);

		if bale ~= nil then
			bale:mountKinematic(self, attachNode, 0, 0, 0, 0, 0, 0);
			bale:setCanBeSold(false);
			bale:setNeedsSaving(false);

			specBaleWrapper.baleToMount = nil;
		else
			specBaleWrapper.baleToMount = {
				serverId = specBaleWrapper.baleGrabber.currentBale,
				linkNode = attachNode,
				trans = {0, 0, 0},
				rot = {0, 0, 0}
			};
		end;

		self:updateWrapNodes(true, false, 0);

		specBaleWrapper.currentWrapper.currentBale = specBaleWrapper.baleGrabber.currentBale;
		specBaleWrapper.baleGrabber.currentBale = nil;

		if specBaleWrapper.currentWrapper.animations.moveToWrapper.animName ~= nil and specBaleWrapper.currentWrapper.animations.moveToWrapper.reverseAfterMove then
			self:playAnimation(specBaleWrapper.currentWrapper.animations.moveToWrapper.animName, -specBaleWrapper.currentWrapper.animations.moveToWrapper.animSpeed, nil, true);
		end;

		specBaleWrapper.baleWrapperState = BaleWrapper.STATE_MOVING_GRABBER_TO_WORK;

	elseif id == BaleWrapper.CHANGE_WRAPPING_START then
		local allowWrappingBale = true;

		if specToggleableBaleWrapper.currentDropModeState == ToggleableBaleWrapper.DROP_MODE_COLLECT then
			local attacherVehicle = self:getAttacherVehicle();

			if attacherVehicle ~= nil then
				specRoundBalerExtension = attacherVehicle.spec_roundBalerExtension;

				if specRoundBalerExtension ~= nil and specRoundBalerExtension.wrappedBaleCount > 0 then
					allowWrappingBale = specRoundBalerExtension.driveOffTimer >= (specRoundBalerExtension.driveOffTimerMax / 2);
				end;
			end;
		end;
			
		if allowWrappingBale then
			specBaleWrapper.baleWrapperState = BaleWrapper.STATE_WRAPPER_WRAPPING_BALE;

			if self.isClient then
				g_soundManager:playSamples(specBaleWrapper.currentWrapper.samples.start);
				g_soundManager:playSamples(specBaleWrapper.currentWrapper.samples.wrap, 0, specBaleWrapper.currentWrapper.samples.start[1]);
			end;

			if specBaleWrapper.currentWrapper.animations.wrapBale.animName ~= nil then
				self:playAnimation(specBaleWrapper.currentWrapper.animations.wrapBale.animName, specBaleWrapper.currentWrapper.animations.wrapBale.animSpeed, nil, true);
			end;
		end;

	elseif id == BaleWrapper.CHANGE_WRAPPING_BALE_FINSIHED then
		if self.isClient then
			g_soundManager:stopSamples(specBaleWrapper.currentWrapper.samples.wrap);
			g_soundManager:stopSamples(specBaleWrapper.currentWrapper.samples.stop);

			if specBaleWrapper.currentWrapper.wrappingSoundEndTime == 1 then
				g_soundManager:playSamples(specBaleWrapper.currentWrapper.samples.stop);
			end;

			g_soundManager:stopSamples(specBaleWrapper.currentWrapper.samples.start);
		end;

		self:updateWrappingState(1, true);

		specBaleWrapper.baleWrapperState = BaleWrapper.STATE_WRAPPER_FINSIHED;

		local bale = NetworkUtil.getObject(specBaleWrapper.currentWrapper.currentBale);
		local baleType = specBaleWrapper.currentWrapper.allowedBaleTypes[specBaleWrapper.currentBaleTypeIndex];
		local skippedWrapping = not bale:getSupportsWrapping() or baleType.skipWrapping;

		if skippedWrapping or bale.wrappingState == 1 then
			self:updateWrappingState(0, true);
		end;

		if not skippedWrapping then
			local animation = specBaleWrapper.currentWrapper.animations.resetWrapping;

			if animation.animName ~= nil then
				self:playAnimation(animation.animName, animation.animSpeed, nil, true);
			end;
		end;

		self:updateConsumable(BaleWrapper.CONSUMABLE_TYPE_NAME, 0);

	elseif id == BaleWrapper.CHANGE_WRAPPER_START_DROP_BALE then
		self:updateWrapNodes(false, false, 0);

		local baleType = specBaleWrapper.currentWrapper.allowedBaleTypes[specBaleWrapper.currentBaleTypeIndex];
        local dropAnimation = baleType.dropAnimations[specBaleWrapper.dropAnimationIndex];
		local dropAnimationName = specBaleWrapper.currentWrapper.animations.dropFromWrapper.animName;

        if dropAnimation ~= nil then
			dropAnimationName = dropAnimation.name;
            specBaleWrapper.currentWrapper.animations.dropFromWrapper.animSpeed = dropAnimation.animSpeed;

            if self.isServer then
                if dropAnimation.liftOnDrop then
                    if self:getIsLowered() then
                        local attacherVehicle = self:getAttacherVehicle();

                        if attacherVehicle ~= nil then
                            attacherVehicle:handleLowerImplementEvent(self);
                        end;
                    end;
                end;
            end;
        end;


		if dropAnimationName ~= nil then
			self:playAnimation(dropAnimationName, specBaleWrapper.currentWrapper.animations.dropFromWrapper.animSpeed, nil, true);
			
			local dropAnimation = self:getAnimationByName(dropAnimationName);
			local bale = NetworkUtil.getObject(specBaleWrapper.currentWrapper.currentBale);

			if not bale:getSupportsWrapping() or baleType.skipWrapping or not specToggleableBaleWrapper.baleWrapperIsActive then
				if self.isClient then
					for sampleNumber = 1, #dropAnimation.samples do
						local sample = dropAnimation.samples[sampleNumber];

						if sample.filename:find("attBalerWrappingStop") then
							if sample.volumeScaleBackup == nil then
								sample.volumeScaleBackup = g_soundManager:getSampleVolumeScale(sample);
							end;

							--## deactive bale foil tear off sound, if bale was not wrapped
							
							g_soundManager:setSampleVolumeScale(sample, 0);
						end;
					end;
				end;
			else
				if self.isClient then
					for sampleNumber = 1, #dropAnimation.samples do
						local sample = dropAnimation.samples[sampleNumber];

						if sample.volumeScaleBackup ~= nil then
							local currentVolumeScale = g_soundManager:getSampleVolumeScale(sample);
	
							if currentVolumeScale ~= sample.volumeScaleBackup then
								--## active bale foil tear off sound, if bale was wrapped and sound was deactivated
								
								g_soundManager:setSampleVolumeScale(sample, sample.volumeScaleBackup);
							end;
						end;
					end;
				end;
			end;
		end;

		specBaleWrapper.dropAnimationIndex = 1;
		specBaleWrapper.baleWrapperState = BaleWrapper.STATE_WRAPPER_DROPPING_BALE;

	elseif id == BaleWrapper.CHANGE_WRAPPER_BALE_DROPPED then
		local bale = NetworkUtil.getObject(specBaleWrapper.currentWrapper.currentBale);

		if bale ~= nil then
			bale:unmountKinematic();
			bale:setNeedsSaving(true);
			bale:setCanBeSold(true);

			local baleType = specBaleWrapper.currentWrapper.allowedBaleTypes[specBaleWrapper.currentBaleTypeIndex];

			if bale:getSupportsWrapping() and not baleType.skipWrapping and specToggleableBaleWrapper.baleWrapperIsActive then
				local stats = g_currentMission:farmStats(self:getOwnerFarmId());
				local total = stats:updateStats("wrappedBales", 1);

				g_achievementManager:tryUnlock("WrappedBales", total);

				if bale.wrappingState < 1 then
					bale:setWrappingState(1);
				end;
			end;
		end;

		specBaleWrapper.lastDroppedBale = bale;
		specBaleWrapper.currentWrapper.currentBale = nil;
		specBaleWrapper.currentWrapper.currentTime = 0;

		if specBaleWrapper.currentWrapper.animations.resetAfterDrop.animName ~= nil then
			local dropAnimationName = specBaleWrapper.currentWrapper.animations.dropFromWrapper.animName;

			if dropAnimationName ~= nil then
				if self:getIsAnimationPlaying(dropAnimationName) then
					self:stopAnimation(dropAnimationName, true);
					self:setAnimationTime(dropAnimationName, 1, true, false);
				end;
			end;

			self:playAnimation(specBaleWrapper.currentWrapper.animations.resetAfterDrop.animName, specBaleWrapper.currentWrapper.animations.resetAfterDrop.animSpeed, nil, true);
		end;

		self:setBaleWrapperType(specBaleWrapper.currentWrapper == specBaleWrapper.roundBaleWrapper, specBaleWrapper.currentBaleTypeIndex);

		specBaleWrapper.baleWrapperState = BaleWrapper.STATE_WRAPPER_RESETTING_PLATFORM;

	elseif id == BaleWrapper.CHANGE_WRAPPER_PLATFORM_RESET then
		self:updateWrappingState(0);
		self:updateWrapNodes(false, true, 0);

		specBaleWrapper.baleWrapperState = BaleWrapper.STATE_NONE;

	elseif id == BaleWrapper.CHANGE_BUTTON_EMPTY then
		assert(self.isServer);

		if specBaleWrapper.baleWrapperState == BaleWrapper.STATE_WRAPPER_FINSIHED then
			g_server:broadcastEvent(BaleWrapperStateEvent.new(self, BaleWrapper.CHANGE_WRAPPER_START_DROP_BALE), true, nil, self);
		end;
	end;

	BaleWrapper.updateActionEvents(self);
end;

function ToggleableBaleWrapper:pickupWrapperBale(superFunc, bale, baleTypeIndex)
    local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
	local specBaleWrapper = self.spec_baleWrapper
	
    if bale:getSupportsWrapping() and specToggleableBaleWrapper.baleWrapperIsActive then
        local baleTypes = bale.isRoundbale and specBaleWrapper.roundBaleWrapper.allowedBaleTypes or specBaleWrapper.squareBaleWrapper.allowedBaleTypes;

        if baleTypes ~= nil and baleTypeIndex ~= nil then
            local baleType = baleTypes[baleTypeIndex];

            if baleType ~= nil then
                if not baleType.skipWrapping and bale.wrappingState < 1 then
                    if baleType.wrapDiffuse ~= nil or baleType.wrapNormal ~= nil then
                        bale:setWrapTextures(baleType.wrapDiffuse, baleType.wrapNormal);
                    end;
                end;
            end;
        end;
    end;

    specBaleWrapper.baleGrabber.balesInTrigger[bale] = nil;

    g_server:broadcastEvent(BaleWrapperStateEvent.new(self, BaleWrapper.CHANGE_GRAB_BALE, NetworkUtil.getObjectId(bale)), true, nil, self);
end;

function ToggleableBaleWrapper:setBaleWrapperState(baleWrapperIsActive, noEventSend)
	local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;

	if baleWrapperIsActive ~= specToggleableBaleWrapper.baleWrapperIsActive then
		if not noEventSend then
			if g_server ~= nil then
				g_server:broadcastEvent(SetBaleWrapperStateEvent.new(self, baleWrapperIsActive), nil, nil, self);
			else
				g_client:getServerConnection():sendEvent(SetBaleWrapperStateEvent.new(self, baleWrapperIsActive));
			end;
		end;

		specToggleableBaleWrapper.baleWrapperIsActive = baleWrapperIsActive;
	end;
end;

function ToggleableBaleWrapper:saveToXMLFile(xmlFile, key)
	local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
	local specBaleWrapper = self.spec_baleWrapper;

	if specBaleWrapper ~= nil then	
		xmlFile:setValue(key .. "#currentDropModeState", specToggleableBaleWrapper.currentDropModeState);
		xmlFile:setValue(key .. "#baleWrapperIsActive", specToggleableBaleWrapper.baleWrapperIsActive);
	end;
end;

function ToggleableBaleWrapper.actionEventEmpty(self, superFunc, actionName, inputValue, callbackState, isAnalog)
    if self.spec_baleWrapper.baleWrapperState == BaleWrapper.STATE_WRAPPER_FINSIHED then
        local dropIsAllowed, warning = self:getIsBaleDropAllowed();

        if dropIsAllowed then
			local attacherVehicle = self:getAttacherVehicle();

			if attacherVehicle ~= nil then
				specRoundBalerExtension = attacherVehicle.spec_roundBalerExtension;

				if specRoundBalerExtension ~= nil then
					specRoundBalerExtension.wrappedBaleCount = 0;
					specRoundBalerExtension.driveOffTimer = 0;

					RoundBalerExtension:resetForNextBale(specRoundBalerExtension, attacherVehicle);
				end;
			end;

            g_client:getServerConnection():sendEvent(BaleWrapperStateEvent.new(self, BaleWrapper.CHANGE_BUTTON_EMPTY));
        elseif warning ~= nil then
            g_currentMission:showBlinkingWarning(warning, 2000);
        end;
    end;
end;

BaleWrapper.actionEventEmpty = Utils.overwrittenFunction(BaleWrapper.actionEventEmpty, ToggleableBaleWrapper.actionEventEmpty);

function ToggleableBaleWrapper:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	if self:getIsActiveForInput(true, false) then
		local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
		local specBaleWrapper = self.spec_baleWrapper;
		
		if specBaleWrapper ~= nil then
			ToggleableBaleWrapper.updateActionEvents(self);

			if specBaleWrapper.baleWrapperState == BaleWrapper.STATE_WRAPPER_FINSIHED then
				local allowDropBale = specToggleableBaleWrapper.currentDropModeState == ToggleableBaleWrapper.DROP_MODE_AUTOMATIC;
				local attacherVehicle = self:getAttacherVehicle();
				local allowForceDropBale = true;

				if attacherVehicle ~= nil then
					specRoundBalerExtension = attacherVehicle.spec_roundBalerExtension;

					if specRoundBalerExtension ~= nil then
						allowForceDropBale = specRoundBalerExtension.driveOffTimer >= specRoundBalerExtension.driveOffTimerMax;
					end;
				end;

				if specToggleableBaleWrapper.currentDropModeState ~= ToggleableBaleWrapper.DROP_MODE_COLLECT and ToggleableBaleWrapper.FORCE_DROP_BALE then
					ToggleableBaleWrapper.FORCE_DROP_BALE = false;
				end;
	
				if ToggleableBaleWrapper.FORCE_DROP_BALE and allowForceDropBale then
					allowDropBale = true;
	
					ToggleableBaleWrapper.FORCE_DROP_BALE = false;
				elseif specToggleableBaleWrapper.currentDropModeState == ToggleableBaleWrapper.DROP_MODE_COLLECT then
					if self.spec_fillUnit:getFillUnitFillLevel(1) == self.spec_fillUnit:getFillUnitCapacity(1) then
						allowDropBale = true;
	
						ToggleableBaleWrapper.FORCE_DROP_BALE = true;
					end;
				end;

				if self.isClient and allowDropBale then
					g_client:getServerConnection():sendEvent(BaleWrapperStateEvent.new(self, BaleWrapper.CHANGE_BUTTON_EMPTY));
				end;
			end;
		end;
	end;
end;

function ToggleableBaleWrapper:updateActionEvents(superFunc)
	local specToggleableBaleWrapper = self.spec_toggleableBaleWrapper;
	local specBaleWrapper = self.spec_baleWrapper;
	
	local toggleBaleWrapperButton = specToggleableBaleWrapper.actionEvents[InputAction.TOGGLE_BALE_WRAPPER_BUTTON];
	local changeDropModeButton = specToggleableBaleWrapper.actionEvents[InputAction.CHANGE_DROP_MODE_BUTTON];

	if toggleBaleWrapperButton ~= nil then
		local currentText = specToggleableBaleWrapper.l10NTexts.BALE_WRAPPER_MODE_DEACTIVATED;

		if specToggleableBaleWrapper.baleWrapperIsActive then
			currentText = specToggleableBaleWrapper.l10NTexts.BALE_WRAPPER_MODE_ACTIVATED;
		end;

		specToggleableBaleWrapper.allowSetBaleWrapperState = true;

		if specBaleWrapper.currentWrapper.currentBale ~= nil then
			local bale = NetworkUtil.getObject(specBaleWrapper.currentWrapper.currentBale);
			
			specToggleableBaleWrapper.allowSetBaleWrapperState = bale:getSupportsWrapping();
		end;
		
		local currentFillType = g_fillTypeManager:getFillTypeByIndex(self:getFillUnitFillType(self.spec_baler.fillUnitIndex)).name;

		if currentFillType ~= nil and currentFillType ~= "UNKNOWN" then
			specToggleableBaleWrapper.allowSetBaleWrapperState = specToggleableBaleWrapper.allowSetBaleWrapperState and not currentFillType:find("DRY") and not currentFillType:find("STRAW");
		end;

		if specToggleableBaleWrapper.baleWrapperIsActive and not specToggleableBaleWrapper.allowSetBaleWrapperState then
			self:setBaleWrapperState(false, false);
		end;

		g_inputBinding:setActionEventActive(toggleBaleWrapperButton.actionEventId, specToggleableBaleWrapper.allowSetBaleWrapperState);
		g_inputBinding:setActionEventTextVisibility(toggleBaleWrapperButton.actionEventId, specToggleableBaleWrapper.allowSetBaleWrapperState);
		g_inputBinding:setActionEventText(toggleBaleWrapperButton.actionEventId, currentText);
	end;

	if changeDropModeButton ~= nil then
		local currentText = specToggleableBaleWrapper.l10NTexts.DROP_MODE_MANUALLY;

		g_inputBinding:setActionEventActive(changeDropModeButton.actionEventId, not specToggleableBaleWrapper.isGoeweilBaler);

		for _, currentDropModeState in pairs({"DROP_MODE_AUTOMATIC", "DROP_MODE_COLLECT"}) do
			if specToggleableBaleWrapper.currentDropModeState == ToggleableBaleWrapper[currentDropModeState] then
				currentText = specToggleableBaleWrapper.l10NTexts[currentDropModeState];
			end;
		end;

		g_inputBinding:setActionEventText(changeDropModeButton.actionEventId, specToggleableBaleWrapper.l10NTexts.DROP_MODE_CURRENT:gsub("currentDropMode", currentText));
	end;
end;

-----------------------------------------------------------------------------
--## Multiplayer Event | Change Drop Mode
-----------------------------------------------------------------------------

SetBaleWrapperDropModeEvent = {};
SetBaleWrapperDropModeEvent_mt = Class(SetBaleWrapperDropModeEvent, Event);

InitEventClass(SetBaleWrapperDropModeEvent, "SetBaleWrapperDropModeEvent");

function SetBaleWrapperDropModeEvent.emptyNew()
    local self = Event.new(SetBaleWrapperDropModeEvent_mt);

    self.className = "SetBaleWrapperDropModeEvent";

    return self;
end;

function SetBaleWrapperDropModeEvent.new(baleWrapper, currentDropModeState)
    local self = SetBaleWrapperDropModeEvent.emptyNew();
	
    self.baleWrapper = baleWrapper;
    self.currentDropModeState = currentDropModeState;
    
	return self;
end;

function SetBaleWrapperDropModeEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.baleWrapper);

    streamWriteInt32(streamId, self.currentDropModeState);
end;

function SetBaleWrapperDropModeEvent:readStream(streamId, connection)
    self.baleWrapper = NetworkUtil.readNodeObject(streamId);
    self.currentDropModeState = streamReadInt32(streamId);

    self:run(connection);
end;

function SetBaleWrapperDropModeEvent:run(connection)	
	if not connection:getIsServer() then
		g_server:broadcastEvent(SetBaleWrapperDropModeEvent.new(self.baleWrapper, self.currentDropModeState), nil, connection, self.baleWrapper);
	end;
	
	if self.baleWrapper ~= nil and self.baleWrapper.setBaleWrapperDropMode ~= nil then	
		self.baleWrapper:setBaleWrapperDropMode(self.currentDropModeState, true);
	end;
end;

-----------------------------------------------------------------------------
--## Multiplayer Event | Turn off/on bale wrapper
-----------------------------------------------------------------------------

SetBaleWrapperStateEvent = {};
SetBaleWrapperStateEvent_mt = Class(SetBaleWrapperStateEvent, Event);

InitEventClass(SetBaleWrapperStateEvent, "SetBaleWrapperStateEvent");

function SetBaleWrapperStateEvent.emptyNew()
	local self = Event.new(SetBaleWrapperStateEvent_mt);
    
	return self;
end;

function SetBaleWrapperStateEvent.new(baleWrapper, baleWrapperIsActive)
	local self = SetBaleWrapperStateEvent.emptyNew();
	
	self.baleWrapper = baleWrapper;
	self.baleWrapperIsActive = baleWrapperIsActive;
	
	return self;
end;

function SetBaleWrapperStateEvent:readStream(streamId, connection)
	self.baleWrapper = NetworkUtil.readNodeObject(streamId);
	self.baleWrapperIsActive = streamReadBool(streamId);
    
	self:run(connection);
end;

function SetBaleWrapperStateEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.baleWrapper);

	streamWriteBool(streamId, self.baleWrapperIsActive);
end;

function SetBaleWrapperStateEvent:run(connection)
	if not connection:getIsServer() then
		g_server:broadcastEvent(SetBaleWrapperStateEvent.new(self.baleWrapper, self.baleWrapperIsActive), nil, connection, self.baleWrapper);
	end;
	
    if self.baleWrapper ~= nil and self.baleWrapper.setBaleWrapperState ~= nil then
        self.baleWrapper:setBaleWrapperState(self.baleWrapperIsActive, true);
	end;
end;