--[[
	StopFullCombine.lua
	
	Autor: 		Ifko[nator]
	Datum: 		28.11.2024
	Version: 	2.1
	
	Changelog:	v1.0 @31.03.2022 - initial implementation in FS 22
				-------------------------------------------------------------------------------------------------------------------------------------
				v1.1 @01.04.2022 - stopped straw swath, when combine is stopped
				-------------------------------------------------------------------------------------------------------------------------------------
				v1.2 @03.04.2022 - pipe will now fold in, when combine is empty
				-------------------------------------------------------------------------------------------------------------------------------------
				v1.3 @11.04.2022 - fix for attached combines
				-------------------------------------------------------------------------------------------------------------------------------------
				v1.4 @28.04.2022 - combines stop now also on dedi servers
				-------------------------------------------------------------------------------------------------------------------------------------
				v1.5 @10.09.2022 - added pipe modes
				-------------------------------------------------------------------------------------------------------------------------------------
				v2.0 @19.11.2024 - convert to FS 25
				-------------------------------------------------------------------------------------------------------------------------------------
				v2.1 @28.11.2024 - fix for following error: FS25_stopFullCombine/StopFullCombine.lua:119: attempt to index nil with 'unloadingStates'
]]

StopFullCombine = {};

StopFullCombine.PIPE_MODE_OFF = 1;
StopFullCombine.PIPE_MODE_MAX = 5;

StopFullCombine.currentModName = "";

for _, mod in pairs(g_modManager.mods) do
	if mod.title:upper() == "STOP FULL COMBINE" or mod.title:upper() == "AUTOMATISCHES STOPPEN FÜR VOLLE MÄHDRESCHER"  then		
		if g_modIsLoaded[tostring(mod.modName)] then	
			StopFullCombine.currentModName = mod.modName;

			break;
		end;
	end;
end;

function StopFullCombine.initSpecialization()
	Vehicle.xmlSchemaSavegame:register(XMLValueType.FLOAT, "vehicles.vehicle(?).stopFullCombine#pipeMode", "Current pipe mode.", 1);
end;

function StopFullCombine.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Combine, specializations);
end;

function StopFullCombine.registerFunctions(vehicleType)
	local newFunctions = {
		"lockCombine",
		"resetTimer",
		"startBeaconLights",
		"stopBeaconLights",
		"switchPipeMode",
		"unfoldPipe",
		"turnOnStrawChopper",
		"moveCutterDown"
	};

	for _, newFunction in ipairs(newFunctions) do
		SpecializationUtil.registerFunction(vehicleType, newFunction, StopFullCombine[newFunction]);
	end;
end;

function StopFullCombine.registerEventListeners(vehicleType)
	local functionNames = {
		"onLoad",
		"onUpdate",
		"onReadStream",
		"onWriteStream",
		"onRegisterActionEvents"
	};

	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, StopFullCombine);
	end;
end;

function StopFullCombine:onLoad(savegame)	
	local specStopFullCombine = self.spec_stopFullCombine;
	local specCombine = self.spec_combine;

	if specCombine.loadingDelay > 0 then
		specCombine.loadingDelay = 0;
	end;

	self:resetTimer(specStopFullCombine);

	specStopFullCombine.stopTimerMax = 5000; --## stop combine for 5 seconds

	specStopFullCombine.needTurnOnBeaconLights = true;
	specStopFullCombine.beaconLightsWasTurnedOn = false;
	specStopFullCombine.allowChangePipeState = false;
	specStopFullCombine.allowTurnOnStrawChopper = false;
	specStopFullCombine.allowMoveCutterDown = false;
	specStopFullCombine.useBunkerText = Utils.getNoNil(getXMLString(self.xmlFile.handle, "vehicle.pipe#pipeInText"), ""):lower():find("bunker");

	specStopFullCombine.pipeMode = StopFullCombine.PIPE_MODE_OFF;

	if savegame ~= nil then
		specStopFullCombine.pipeMode = savegame.xmlFile:getValue(savegame.key .. ".stopFullCombine#pipeMode", StopFullCombine.PIPE_MODE_OFF);	
	end;

	self:switchPipeMode(specStopFullCombine.pipeMode, false);

	specStopFullCombine.l10NTexts = {};

	local l10NTexts = {
		"PIPE_MODE_OFF",
		"PIPE_MODE_X",
		"BUNKER_MODE_OFF",
		"BUNKER_MODE_X",
		"input_SET_PIPE_BUNKER_MODE_BUTTON"
	};

	for _, l10NText in pairs(l10NTexts) do
		specStopFullCombine.l10NTexts[l10NText] = g_i18n:getText(l10NText, StopFullCombine.currentModName);
	end;
end;

function StopFullCombine:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	local specCombine = self.spec_combine;
	local specPipe = self.spec_pipe;
	
	if self:getIsActive() and specPipe ~= nil and specPipe.hasMovablePipe and #specPipe.unloadingStates == 0 and not specCombine.isBufferCombine then
		local specStopFullCombine = self.spec_stopFullCombine;
		local specLights = self.spec_lights;
		local specDrivable = self.spec_drivable;
		local isMotorStarted = self.getIsMotorStarted ~= nil and self:getIsMotorStarted();
		local controlledTractor = nil;
		
		if self.getAttacherVehicle ~= nil then
			controlledTractor = self:getAttacherVehicle();
		end;

		if controlledTractor ~= nil then
			specDrivable = controlledTractor.spec_drivable;
			isMotorStarted = controlledTractor.getIsMotorStarted ~= nil and controlledTractor:getIsMotorStarted();
		end;

		if specDrivable ~= nil and specCombine ~= nil and isMotorStarted then
			local maxFillLevelPercentageToUnfoldPipe = specStopFullCombine.pipeMode > 1 and 1 or 0;

			if specStopFullCombine.pipeMode == 2 then
				maxFillLevelPercentageToUnfoldPipe = 0.7;
			elseif specStopFullCombine.pipeMode == 3 then
				maxFillLevelPercentageToUnfoldPipe = 0.8;
			elseif specStopFullCombine.pipeMode == 4 then
				maxFillLevelPercentageToUnfoldPipe = 0.9;
			end;

			if maxFillLevelPercentageToUnfoldPipe > 0 then
				if self:getFillUnitFillLevelPercentage(specCombine.fillUnitIndex) >= maxFillLevelPercentageToUnfoldPipe then
					self:unfoldPipe(self.spec_pipe, specStopFullCombine);
				end;
			end;

			if self:getFillUnitFillLevelPercentage(specCombine.fillUnitIndex) >= 0.9 then
				self:startBeaconLights(specStopFullCombine.needTurnOnBeaconLights, specStopFullCombine, controlledTractor);
			else
				self:stopBeaconLights(specStopFullCombine.beaconLightsWasTurnedOn, specStopFullCombine, controlledTractor);
			end;
			
			if self:getFillUnitFillLevelPercentage(specCombine.fillUnitIndex) >= 1 then
				if specStopFullCombine.stopTimer < specStopFullCombine.stopTimerMax then
					self:lockCombine(self.spec_wheels, controlledTractor);

					specStopFullCombine.stopTimer = specStopFullCombine.stopTimer + dt;
				end;

				if specCombine.isSwathActive then
					self:turnOnStrawChopper(specStopFullCombine.allowTurnOnStrawChopper, specStopFullCombine);
				end;

				--## move cutter down, when combine is full. In the base game it will be raised automatically
				self:moveCutterDown(specStopFullCombine.allowMoveCutterDown, specStopFullCombine);
			elseif specStopFullCombine.pipeMode > 1 and self:getFillUnitFreeCapacity(specCombine.fillUnitIndex) == self:getFillUnitCapacity(specCombine.fillUnitIndex) then
				if specStopFullCombine.allowChangePipeState then
					--## fold pipe in, when empty

					self:setPipeState(1);

					specStopFullCombine.allowChangePipeState = false;
				end;
			else
				if specStopFullCombine.stopTimer > 0 then
					self:resetTimer(specStopFullCombine);
				end;

				specStopFullCombine.allowTurnOnStrawChopper = specCombine.isSwathActive;

				specStopFullCombine.allowMoveCutterDown = specCombine.numAttachedCutters > 0;
			end;
			
			local setPipeBunkerModeButton = specStopFullCombine.actionEvents[InputAction.SET_PIPE_BUNKER_MODE_BUTTON];

			if setPipeBunkerModeButton ~= nil then
				StopFullCombine.updateActionEventText(setPipeBunkerModeButton.actionEventId, specStopFullCombine);
			end;
		end;
	end;
end;

function StopFullCombine:onRegisterActionEvents(isActiveForInput)
	if self.isClient then
        local specStopFullCombine = self.spec_stopFullCombine;
		local specPipe = self.spec_pipe;

		self:clearActionEventsTable(specStopFullCombine.actionEvents);

		if self:getIsActiveForInput(true, false) and specPipe ~= nil and specPipe.hasMovablePipe and #specPipe.unloadingStates == 0 and not self.spec_combine.isBufferCombine then
			local _, actionEventId = self:addActionEvent(specStopFullCombine.actionEvents, InputAction.SET_PIPE_BUNKER_MODE_BUTTON, self, StopFullCombine.actionEventSetPipeMode, false, true, false, true, nil);

			g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_VERY_HIGH);
			g_inputBinding:setActionEventTextVisibility(actionEventId, true);
			g_inputBinding:setActionEventActive(actionEventId, true);

			StopFullCombine.updateActionEventText(actionEventId, specStopFullCombine);
		end;
	end;
end;

function StopFullCombine.updateActionEventText(actionEventId, specStopFullCombine)
	if actionEventId ~= nil then	
		local currentText = specStopFullCombine.useBunkerText and specStopFullCombine.l10NTexts.BUNKER_MODE_OFF or specStopFullCombine.l10NTexts.PIPE_MODE_OFF;

		if specStopFullCombine.pipeMode == 2 then
			currentText = specStopFullCombine.useBunkerText and specStopFullCombine.l10NTexts.BUNKER_MODE_X:format("70 %") or specStopFullCombine.l10NTexts.PIPE_MODE_X:format("70 %");
		elseif specStopFullCombine.pipeMode == 3 then
			currentText = specStopFullCombine.useBunkerText and specStopFullCombine.l10NTexts.BUNKER_MODE_X:format("80 %") or specStopFullCombine.l10NTexts.PIPE_MODE_X:format("80 %");
		elseif specStopFullCombine.pipeMode == 4 then
			currentText = specStopFullCombine.useBunkerText and specStopFullCombine.l10NTexts.BUNKER_MODE_X:format("90 %") or specStopFullCombine.l10NTexts.PIPE_MODE_X:format("90 %");
		elseif specStopFullCombine.pipeMode == 5 then
			currentText = specStopFullCombine.useBunkerText and specStopFullCombine.l10NTexts.BUNKER_MODE_X:format("100 %") or specStopFullCombine.l10NTexts.PIPE_MODE_X:format("100 %");
		end;

		g_inputBinding:setActionEventText(actionEventId, currentText);
	end;
end;

function StopFullCombine.actionEventSetPipeMode(self, actionName, inputValue, callbackState, isAnalog)
	local specStopFullCombine = self.spec_stopFullCombine;

	if specStopFullCombine.useBunkerText then
		if specStopFullCombine.pipeMode < StopFullCombine.PIPE_MODE_MAX then
			specStopFullCombine.pipeMode = StopFullCombine.PIPE_MODE_MAX;
		else
			specStopFullCombine.pipeMode = StopFullCombine.PIPE_MODE_OFF;
		end;
	else
		if specStopFullCombine.pipeMode < StopFullCombine.PIPE_MODE_MAX then
			specStopFullCombine.pipeMode = specStopFullCombine.pipeMode + 1;
		else
			specStopFullCombine.pipeMode = StopFullCombine.PIPE_MODE_OFF;
		end;
	end;

	self:switchPipeMode(specStopFullCombine.pipeMode, false);
end;

function StopFullCombine:switchPipeMode(pipeMode, noEventSend)
	local specStopFullCombine = self.spec_stopFullCombine;

	if pipeMode ~= specStopFullCombine.pipeMode then
		if not noEventSend then
			if g_server ~= nil then
				g_server:broadcastEvent(StopFullCombineEvent.new(self, pipeMode), nil, nil, self);
			else
				g_client:getServerConnection():sendEvent(StopFullCombineEvent.new(self, pipeMode));
			end;
		end;

		specStopFullCombine.pipeMode = pipeMode;
	end;
end;

function StopFullCombine:saveToXMLFile(xmlFile, key)
	local specStopFullCombine = self.spec_stopFullCombine;
	local specPipe = self.spec_pipe;

	if specStopFullCombine.pipeMode > 1 and specPipe ~= nil and specPipe.hasMovablePipe and #specPipe.unloadingStates == 0 and not self.spec_combine.isBufferCombine then
		xmlFile:setValue(key .. "#pipeMode", specStopFullCombine.pipeMode);
	end;
end;

function StopFullCombine:onReadStream(streamId, connection)
	self.spec_stopFullCombine.pipeMode = streamReadFloat32(streamId);
end;

function StopFullCombine:onWriteStream(streamId, connection)
	streamWriteFloat32(streamId, self.spec_stopFullCombine.pipeMode);
end;

function StopFullCombine:resetTimer(specStopFullCombine)
	specStopFullCombine.stopTimer = 0;
end;

function StopFullCombine:lockCombine(specWheels, controlledTractor)
	if self.isServer then
		for _, wheel in pairs(specWheels.wheels) do
			setWheelShapeProps(wheel.node, wheel.physics.wheelShape, 0, math.huge, 0, 0);
		end;

		if controlledTractor ~= nil then
			for _, wheel in pairs(controlledTractor.spec_wheels.wheels) do
				setWheelShapeProps(wheel.node, wheel.physics.wheelShape, 0, math.huge, 0, 0);
			end;
		end;
	end;

	self:setBrakeLightsVisibility(true);

	if self.getCruiseControlState ~= nil and self:getCruiseControlState() ~= Drivable.CRUISECONTROL_STATE_OFF then
		self:setCruiseControlState(Drivable.CRUISECONTROL_STATE_OFF);
	end;

	if controlledTractor ~= nil and controlledTractor.getCruiseControlState ~= nil and controlledTractor:getCruiseControlState() ~= Drivable.CRUISECONTROL_STATE_OFF then
		controlledTractor:setCruiseControlState(Drivable.CRUISECONTROL_STATE_OFF);
	end;
end;

function StopFullCombine:unfoldPipe(specPipe, specStopFullCombine)
	if specPipe ~= nil and specPipe.hasMovablePipe then
		if specPipe.targetState == 1 then
			--## unfold pipe, when limit is reached

			self:setPipeState(next(specPipe.unloadingStates));
		end;

		specStopFullCombine.allowChangePipeState = true;
	end;
end;

function StopFullCombine:startBeaconLights(needTurnOnBeaconLights, specStopFullCombine, controlledTractor)
	if needTurnOnBeaconLights then
		local specLights = self.spec_lights;
		
		if specLights ~= nil and #specLights.beaconLights > 0 then
			self:setBeaconLightsVisibility(true, true, nil);
		end;

		if controlledTractor ~= nil then
			specLights = controlledTractor.spec_lights;

			if specLights ~= nil and #specLights.beaconLights > 0 then
				controlledTractor:setBeaconLightsVisibility(true, true, nil);
			end;
		end;

		specStopFullCombine.needTurnOnBeaconLights = false;
		specStopFullCombine.beaconLightsWasTurnedOn = true;
	end;
end;

function StopFullCombine:stopBeaconLights(beaconLightsWasTurnedOn, specStopFullCombine, controlledTractor)
	if beaconLightsWasTurnedOn then
		local specLights = self.spec_lights;
		
		if specLights ~= nil and #specLights.beaconLights > 0 then
			self:setBeaconLightsVisibility(false, true, nil);
		end;

		if controlledTractor ~= nil then
			specLights = controlledTractor.spec_lights;

			if specLights ~= nil and #specLights.beaconLights > 0 then
				controlledTractor:setBeaconLightsVisibility(false, true, nil);
			end;
		end;

		specStopFullCombine.needTurnOnBeaconLights = true;
		specStopFullCombine.beaconLightsWasTurnedOn = false;
	end;
end;

function StopFullCombine:turnOnStrawChopper(allowTurnOnStrawChopper, specStopFullCombine)
	if allowTurnOnStrawChopper then
		self:setIsSwathActive(false, true, true);

		specStopFullCombine.allowTurnOnStrawChopper = false;

		self:setIsSwathActive(true, true, true);
	end;
end;

function StopFullCombine:moveCutterDown(allowMoveCutterDown, specStopFullCombine)
	for _, cutter in pairs(self.spec_combine.attachedCutters) do
		if allowMoveCutterDown and cutter ~= self then
			local jointDescIndex = self:getAttacherJointIndexFromObject(cutter);

			self:setJointMoveDown(jointDescIndex, true, true);

			specStopFullCombine.allowMoveCutterDown = false;
		end;
	end;
end;

StopFullCombineEvent = {};
StopFullCombineEvent_mt = Class(StopFullCombineEvent, Event);

InitEventClass(StopFullCombineEvent, "StopFullCombineEvent");

function StopFullCombineEvent.emptyNew()
	local self = Event.new(StopFullCombineEvent_mt);
    
	return self;
end;

function StopFullCombineEvent.new(combine, pipeMode)
	local self = BaleCounterEvent.emptyNew();
	
	self.combine = combine;
	self.pipeMode = pipeMode;
	
	return self;
end;

function StopFullCombineEvent:readStream(streamId, connection)
	self.combine = NetworkUtil.readNodeObject(streamId);
	self.pipeMode = streamReadInt32(streamId);

	self:run(connection);
end;

function StopFullCombineEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.combine);

	streamWriteInt32(streamId, self.pipeMode);
end;

function StopFullCombineEvent:run(connection)
	if not connection:getIsServer() then
		g_server:broadcastEvent(StopFullCombineEvent.new(self.combine, self.pipeMode), nil, connection, self.combine);
	end;

    if self.combine ~= nil then
        self.combine:switchPipeMode(self.pipeMode, true);
	end;
end;